<?php

/**
 * Escapes text for HTML output, allowing certain tags.
 *
 * @param string $text        Content to escape.
 * @param array  $allowedtags Allowed tags, see description.
 * @param string $context     kses context to use, {@see wp_kses_allowed_html}.
 *
 * @return string             Escaped string for output into HTML context.
 */
function inspiry_whitelist_html( $text, $allowedtags = array(), $context = '' ) {

	$actually_allowed = array();
	$default_list     = wp_kses_allowed_html( $context );

	// Split comma-separated string.
	if ( is_string( $allowedtags ) ) {
		$allowedtags = array_map( 'trim', explode( ',', $allowedtags ) );
	}

	foreach ( $allowedtags as $key => $tag ) {
		if ( is_array( $tag ) && is_string( $key ) ) {
			// kses-formatted of `'element' => [ 'attr' => true ]`.
			// `$tag` is actually the attrs, and `$key` is the tag name.
			$actually_allowed[ $key ] = $tag;
			continue;
		}

		if ( ! is_string( $tag ) ) {
			// Not concise form, what even is this?
			_doing_it_wrong( 'inspiry_whitelist_html', '$allowedtags must consist of strings or kses-style arrays', '4.7.5' );
			continue;
		}

		// Grab default attributes for the tag.
		$attrs = array();
		if ( isset( $default_list[ $tag ] ) ) {
			$attrs = $default_list[ $tag ];
		}

		// Add to allowed list.
		$actually_allowed[ $tag ] = $attrs;
	}

	// Do the sanitization dance.
	$sanitized = wp_kses( $text, $actually_allowed );

	/**
	 * Filter a string to be output into HTML, allowing some tags.
	 *
	 * @param string $sanitized   The text after it has been escaped.
	 * @param string $text        The text before it has been escaped.
	 * @param string $allowedtags Tags requested to whitelist.
	 * @param string $context     Original kses context to use, {@see wp_kses_allowed_html}.
	 */
	return apply_filters( 'inspiry_whitelist_html', $sanitized, $text, $allowedtags, $context );
}

/**
 * Escapes text for HTML output, allowing certain tags, then outputs.
 *
 * @see inspiry_whitelist_html
 *
 * @param string $text        Content to escape.
 * @param array  $allowedtags Allowed tags, {@see inspiry_whitelist_html}.
 * @param string $context     kses context to use, {@see wp_kses_allowed_html}.
 *
 * @return string             Escaped string for output into HTML context.
 */
function inspiry_print_whitelist_html( $text, $allowedtags = array(), $context = '' ) {
	echo inspiry_whitelist_html( $text, $allowedtags, $context );
}
