<?php
/**
 * Utility Functions
 */
if ( ! function_exists( 'inspiry_section_heading' ) ) {
	/**
	 * @param string $title
	 * @param string $description
	 */
	function inspiry_section_heading( $title = '', $description = '' ) {

		if ( ! empty( $title ) || ! empty( $description ) ) {

			echo '<div class="section-heading">';

			if ( ! empty( $title ) ) {
				echo '<h2>' . $title . '</h2>';
			}

			if ( ! empty( $description ) ) {
				echo '<p>' . $description . '</p>';
			}

			echo '</div>';
		}
	}
}

if ( ! function_exists( 'inspiry_header_banner_image' ) ) {
	/**
	 * Display Header Banner Image With Settings
	 */
	function inspiry_header_banner_image() {

		if ( is_home() ) {
			$the_id          = get_option( 'page_for_posts' );
			$banner_image_id = get_post_meta( $the_id, 'Inspiry_Tours_banner_image', true );
		} elseif ( is_singular( 'tour' ) ) {
			$banner_image_id = get_post_meta( get_the_ID(), 'Inspiry_Tours_banner_image', true );
			if ( empty( $banner_image_id ) ) {
				$banner_image_id = get_post_thumbnail_id();
			}
		} elseif ( is_tax( 'tour-type' ) ) {
			$banner_image_id = get_term_meta( get_queried_object()->term_id, 'featured_image', true );
		} else {
			$banner_image_id = get_post_meta( get_the_ID(), 'Inspiry_Tours_banner_image', true );
		}

		if ( ! empty( $banner_image_id ) ) {
			$background_url = wp_get_attachment_image_url( $banner_image_id, 'full', false );
		} else {
			$background_url = get_theme_mod( 'inspiry_shop_banner_image_url' );
			if ( ! is_woocommerce_activated() || ! ( is_shop() || is_product() ) || empty( $background_url ) ) {
				$background_url = get_theme_mod( 'inspiry_banner_image_url' );
			}
		}

		if( ! empty( $background_url ) ) {
			return 'background-image: url(' . inspiry_filter_protocol( esc_url( $background_url ) ) . ');';
		}
	}
}

if ( ! function_exists( 'inspiry_page_banner_tile' ) ) {
	/**
	 * @param $title
	 *
	 * @return string
	 */
	function inspiry_page_banner_tile( $title ) {

		if ( is_home() ) {
			$the_id = get_option( 'page_for_posts' );
		} else {
			$the_id = get_the_ID();
		}

		$custom_title = get_post_meta( $the_id, 'Inspiry_Tours_page_title', true );

		return ( ! empty( $custom_title ) ) ? $custom_title : $title;
	}

	add_filter( 'inspiry_page_banner_tile', 'inspiry_page_banner_tile' );
}

if ( ! function_exists( 'inspiry_blog_title' ) ) {
	/**
	 * Display blog page title
	 */
	function inspiry_blog_title() {

		$blog_title   = esc_html__( 'News', 'inspiry-tourpress' );
		$blog_page_id = get_option( 'page_for_posts' );
		if ( $blog_page_id ) {
			$blog_title = get_the_title( $blog_page_id );
		}

		return $blog_title;
	}

}

if ( ! function_exists( 'inspiry_taxonomy_options' ) ) {
	/**
	 * Display a list of options from a taxonomy
	 * Also make an option selected if given value match term slug
	 *
	 * @param $taxonomy
	 * @param string $value
	 *
	 * @return void
	 */
	function inspiry_taxonomy_options( $taxonomy, $value = '' ) {

		if ( ! empty( $taxonomy ) ) {

			$terms = get_terms( $taxonomy, array(
				'orderby'    => 'count',
				'hide_empty' => 0,
			) );

			if ( ! empty( $terms ) && ! is_wp_error( $terms ) ){

				$selected = '';
				$months = array();

				if ( $taxonomy == 'tour-month' ) {

					foreach ( $terms as $term ) {
						$months[ $term->slug ] = $term->name;
					}

					// Added Support for Simple Custom Post Order Plugin
					( ! class_exists( 'SCPO_Engine' ) ) ? usort( $months, 'inspiry_compare_months' ) : '';

					foreach ( $months as $month ) {

						if ( isset( $_GET[ $value ] ) && strtolower( $month ) == $_GET[ $value ] ) {
							$selected = 'selected';
						}

						echo '<option value="' . esc_attr( strtolower( $month ) ) . '"' . esc_attr( $selected ) . '>' . esc_html( $month ) . '</option>';

						$selected = '';
					}

				} else {
					foreach ( $terms as $term ) {

						if ( isset( $_GET[ $value ] ) && $term->slug == $_GET[ $value ] ) {
							$selected = 'selected';
						}

						echo '<option value="' . esc_attr( $term->slug ) . '"' . esc_attr( $selected ) . '>' . esc_html( $term->name ) . '</option>';

						$selected = '';
					}
				}
			}
		}
	}
}

if( ! function_exists( 'inspiry_compare_months' ) ) {
	/**
	 * Compare months based on their index/number.
	 *
	 * @param $month_1
	 * @param $month_2
	 *
	 * @return mixed // number of months pair calculation.
	 */
	function inspiry_compare_months( $month_1, $month_2 ) {

		$monthA = date_parse( $month_1 );
		$monthB = date_parse( $month_2 );

		return $monthA["month"] - $monthB["month"];
	}
}

if ( ! function_exists( 'inspiry_contact_social_profiles' ) ) {
	/**
	 * Check if any social profile link is set on the contact page
	 *
	 * @param $meta_data
	 *
	 * @return bool
	 */
	function inspiry_contact_social_profiles( $meta_data ) {

		if (
			! empty( $meta_data['Inspiry_Tours_facebook_profile'] )
			|| ! empty( $meta_data['Inspiry_Tours_google_profile'] )
			|| ! empty( $meta_data['Inspiry_Tours_twitter_profile'] )
			|| ! empty( $meta_data['Inspiry_Tours_instagram_profile'] )
			|| ! empty( $meta_data['Inspiry_Tours_pinterest_profile'] )

		) {
			return true;
		}

		return false;
	}
}

if ( ! function_exists( 'inspiry_image_placeholder' ) ) {
	/**
	 * Return or display the placeholder image url based
	 * on given parameters
	 *
	 * @param array $dimensions
	 * @param string $text
	 * @param int $size
	 * @param bool $echo
	 *
	 * @return string
	 */
	function inspiry_image_placeholder(
		$dimensions = array(
			150,
			150
		), $text = '150&#215;150', $size = 10, $echo = false
	) {

		$image_url = 'https://placeholdit.imgix.net/~text?txtsize=' . intval( $size )
		             . '&txt=' . sanitize_text_field( $text )
		             . '&w=' . intval( $dimensions[0] ) . '&h=' . intval( $dimensions[1] );

		if ( ! $echo ) {
			return esc_url( $image_url );
		}

		echo esc_url( $image_url );
	}
}

if ( ! function_exists( 'inspiry_body_classes' ) ) {
	/**
	 * Add custom classes to body tag
	 *
	 * @param $classes
	 *
	 * @return mixed
	 */
	function inspiry_body_classes( $classes ) {

		if ( ( ! is_page_template( 'page-templates/home.php' ) ) && ( is_woocommerce_activated() && ! is_woocommerce() ) ) {
			$classes[] = 'grey-bg';
		}

		$search_page_id = get_theme_mod( 'inspiry_search_page', 0 );
		if ( 0 != $search_page_id && is_page_template( 'page-templates/tours-search.php' ) ) {
			$classes[] = 'search-enabled';
		}

		return $classes;
	}

	add_filter( 'body_class', 'inspiry_body_classes' );
}

if ( ! function_exists( 'inspiry_no_posts' ) ) {
	/**
	 * No posts found notice
	 *
	 * @param $posts // name of post type
	 */
	function inspiry_no_posts( $posts = 'Posts' ) {
		echo sprintf( '<div class="col-md-12"><h4 class="no-posts-found">' . esc_html__( 'No %s Found!', 'inspiry-tourpress' ) . '</h4></div>', $posts );
	}
}

if ( ! function_exists( 'inspiry_get_attachment' ) ) {
	/**
	 * @param $attachment_id
	 *
	 * @return array
	 */
	function inspiry_get_attachment( $attachment_id ) {

		$attachment = get_post( $attachment_id );

		return array(
			'alt'         => get_post_meta( $attachment->ID, '_wp_attachment_image_alt', true ),
			'caption'     => $attachment->post_excerpt,
			'description' => $attachment->post_content,
			'href'        => get_permalink( $attachment->ID ),
			'src'         => $attachment->guid,
			'title'       => $attachment->post_title
		);
	}
}

if ( ! function_exists( 'inspiry_pagination' ) ) {
	/**
	 * Output pagination
	 *
	 * @param $query
	 */
	function inspiry_pagination( $query = '' ) {

		if ( empty( $query ) ) {
			global $wp_query;
			$query = $wp_query;
		}


		$paged          = ( is_front_page() ) ? get_query_var( 'page' ) : get_query_var( 'paged' );
		$big            = 999999999; // need an unlikely integer
		$paginate_links = paginate_links( array(
			'base'      => str_replace( $big, '%#%', esc_url( get_pagenum_link( $big ) ) ),
			'format'    => '?paged=%#%',
			'prev_text' => esc_html__( 'Previous', 'inspiry-tourpress' ),
			'next_text' => esc_html__( 'Next', 'inspiry-tourpress' ),
			'current'   => max( 1, $paged ),
			'total'     => $query->max_num_pages,
		) );

		if ( $paginate_links ) {

			echo "<div class='pagination'><div class='nav-links'>";
			echo wp_kses(
				$paginate_links,
				array(
					'span' => array(
						'aria-current' => array(),
						'class'        => array()
					),
					'a'    => array(
						'href'  => array(),
						'class' => array()
					)
				)
			);
			echo "</div></div>";
		}

	}

}

if ( ! function_exists( 'inspiry_widget_wrap_allowed_tags' ) ) {
	/**
	 * Return allowed tags for the TourPress widget wrapper
	 *
	 * @return array
	 */
	function inspiry_widget_wrap_allowed_tags() {
		return array(
			'section' => array(
				'class' => array(),
				'id'    => array()
			)
		);
	}
}

if ( ! function_exists( 'inspiry_widget_heading_allowed_tags' ) ) {
	/**
	 * Return allowed tags for the TourPress widget heading
	 *
	 * @return array
	 */
	function inspiry_widget_heading_allowed_tags() {
		return array(
			'h2' => array(
				'class' => array()
			),
			'h3' => array(
				'class' => array()
			)
		);
	}
}


if ( ! function_exists( 'inspiry_log' ) ) {
	/**
	 * Function to help in debugging
	 *
	 * @param $message
	 */
	function inspiry_log( $message ) {
		if ( WP_DEBUG === true ) {
			if ( is_array( $message ) || is_object( $message ) ) {
				error_log( print_r( $message, true ) );
			} else {
				error_log( $message );
			}
		}
	}
}

if ( ! function_exists( 'inspiry_get_first_category' ) ) {
	/**
	 * Return post first category
	 *
	 * @param $post_id
	 *
	 * @return string
	 */
	function inspiry_get_first_category( $post_id ) {

		if( empty( $post_id ) ) {
			$post_id = get_the_ID();
		}

		$categories = get_the_category( $post_id );

		if ( empty( $categories ) ) { return; }

		$cat_name   = $categories[0]->name;
		$cat_link   = get_term_link( $categories[0]->term_id );

		return ' <a href="' . esc_url( $cat_link ) . '">' . esc_html( $cat_name ) . '</a>';
	}
}

if ( ! function_exists( 'inspiry_exclude_from_search' ) ) {

	function inspiry_exclude_from_search( $query ) {

		// check is search result query
		if ( ! is_admin() && $query->is_main_query() && $query->is_search() ) {

			$post_type_to_remove = 'product';
			// get all searchable post types
			$searchable_post_types = get_post_types( array( 'exclude_from_search' => false ) );

			// make sure you got the proper results, and that your post type is in the results
			if ( is_array( $searchable_post_types ) && in_array( $post_type_to_remove, $searchable_post_types ) ) {

				// remove the post type from the array
				unset( $searchable_post_types[ $post_type_to_remove ] );

				// set the query to the remaining searchable post types
				$query->set( 'post_type', $searchable_post_types );
			}
		}
	}

	add_action( 'pre_get_posts', 'inspiry_exclude_from_search' );
}

if ( ! function_exists( 'inspiry_hex_to_rgba' ) ) {

	/**
	 * Convert hexdec color string to rgb(a) string
	 */
	function inspiry_hex_to_rgba( $color, $opacity = false ) {

		$default = '';

		// Return default if no color provided
		if ( empty( $color ) ) {
			return $default;
		}

		// Sanitize $color if "#" is provided
		if ( $color[0] == '#' ) {
			$color = substr( $color, 1 );
		}

		// Check if color has 6 or 3 characters and get values
		if ( strlen( $color ) == 6 ) {
			$hex = array( $color[0] . $color[1], $color[2] . $color[3], $color[4] . $color[5] );
		} elseif ( strlen( $color ) == 3 ) {
			$hex = array( $color[0] . $color[0], $color[1] . $color[1], $color[2] . $color[2] );
		} else {
			return $default;
		}

		// Convert hexadec to rgb
		$rgb = array_map( 'hexdec', $hex );

		// Check if opacity is set(rgba or rgb)
		if ( $opacity ) {
			if ( abs( $opacity ) > 1 ) {
				$opacity = 1.0;
			}
			$output = 'rgba(' . implode( ",", $rgb ) . ',' . $opacity . ')';
		} else {
			$output = 'rgb(' . implode( ",", $rgb ) . ')';
		}

		// Return rgb(a) color string
		return $output;
	}
}

if ( ! function_exists( 'inspiry_update_taxonomy_archive_pagination' ) ) {
	/**
	 * Update Taxonomy Pagination Based on Number of Properties Provided in Theme Options
	 */
	function inspiry_update_taxonomy_archive_pagination( $query ) {
		// Avoiding conflict at backend
		if ( is_admin() ) return;
		
		// Detecting the taxonomies
		if ( is_tax( 'tour-destination' )
		|| is_tax( 'tour-type' )
		|| is_tax( 'tour-month' ) 
		|| is_post_type_archive('tour')
		) {
			if ( $query->is_main_query() ) {
				// number of tours to display per page
				$number_of_tours = intval( get_theme_mod( 'inspiry_tax_archive_tours' ) );
				$number_of_tours = ( $number_of_tours ) ? $number_of_tours : 6;
				if ( ! $number_of_tours ) {
					$number_of_tours = 6;
				}
				$query->set( 'posts_per_page', $number_of_tours );
			}
		}
	}

	add_action( 'pre_get_posts', 'inspiry_update_taxonomy_archive_pagination' );
}

if ( ! function_exists( 'inspiry_get_terms_array' ) ) {
	/**
	 * Returns terms array for a given taxonomy containing key(slug) value(name) pair
	 *
	 * @param $tax_name
	 * @param $terms_array
	 */
	function inspiry_get_terms_array( $tax_name, &$terms_array ) {
		$tax_terms = get_terms( array(
			'taxonomy'   => $tax_name,
			'hide_empty' => false,
		) );
		inspiry_add_term_children( 0, $tax_terms, $terms_array );
	}
}

if ( ! function_exists( 'inspiry_add_term_children' ) ) :
	/**
	 * A recursive function to add children terms to given array
	 *
	 * @param $parent_id
	 * @param $tax_terms
	 * @param $terms_array
	 * @param string $prefix
	 */
	function inspiry_add_term_children( $parent_id, $tax_terms, &$terms_array, $prefix = '' ) {
		if ( ! empty( $tax_terms ) && ! is_wp_error( $tax_terms ) ) {
			foreach ( $tax_terms as $term ) {
				if ( $term->parent == $parent_id ) {
					$terms_array[ $term->slug ] = $prefix . $term->name;
					inspiry_add_term_children( $term->term_id, $tax_terms, $terms_array, $prefix . '- ' );
				}
			}
		}
	}
endif;