<?php
/**
 * Tour CPT Functions
 */
if ( ! function_exists( 'inspiry_icon' ) ) {
	/**
	 * Display icon
	 *
	 * @param $icon
	 * @param $label
	 * @param $force_label
	 */
	function inspiry_icon( $icon, $label = '', $force_label = null ) {

		if ( ( $force_label && ! empty( $label ) ) || ! $force_label ) {
			if ( ! empty( $icon ) ) {

				$icon_path = 'tour/';
				if( is_singular( 'tour' ) ) {
					$icon_path = 'tour/single/';
				}

				$icon = $icon_path . $icon;
				?>
				<span class="tour-meta-icon">
				    <span><?php echo esc_html( $label ); ?></span>
					<?php include get_theme_file_path( '/img/svg/' . $icon . '.svg' ); ?>
				</span>
				<?php
			}
		}
	}
}

if ( ! function_exists( 'inspiry_tours_search' ) ) {
	/**
	 * @param $search_args
	 *
	 * @return array search query
	 */
	function inspiry_tours_search( $search_args ) {

		$tax_query = $meta_query = array();

		// tour destination
		if ( isset( $_GET['destination'] ) && ! empty( $_GET['destination'] ) ) {
			$tax_query[] = array(
				'taxonomy' => 'tour-destination',
				'field'    => 'slug',
				'terms'    => esc_html( $_GET['destination'] )
			);
		}

		// tour type
		if ( isset( $_GET['type'] ) && ! empty( $_GET['type'] ) ) {
			$tax_query[] = array(
				'taxonomy' => 'tour-type',
				'field'    => 'slug',
				'terms'    => esc_html( $_GET['type'] )
			);
		}

		// tour month
		if ( isset( $_GET['month'] ) && ! empty( $_GET['month'] ) ) {
			$month_selected = get_term_by('name', esc_html( $_GET['month'] ), 'tour-month');
			$tax_query[] = array(
				'taxonomy' => 'tour-month',
				'field'    => 'slug',
				'terms'    => esc_html( $month_selected->slug )
			);
		}

		// tour maximum price
		if ( isset( $_GET['max_price'] ) && ! empty( intval( $_GET['max_price'] ) ) ) {
			$meta_query[] = array(
				'relation' => 'OR',
				array(
					'key'     => 'inspiry_tour_price',
					'value'   => intval( $_GET['max_price'] ),
					'type'    => 'numeric',
					'compare' => '<=',
				),
				array(
					'key'     => 'inspiry_tour_price_dis',
					'value'   => intval( $_GET['max_price'] ),
					'type'    => 'numeric',
					'compare' => '<=',
				),
			);
		}

		if ( count( $tax_query ) > 1 ) {
			$tax_query['relation'] = 'AND';
		}

		$search_args['tax_query']  = $tax_query;
		$search_args['meta_query'] = $meta_query;

		return $search_args;
	}

	add_filter( 'inspiry_tours_search_args', 'inspiry_tours_search' );
}

if ( ! function_exists( 'inspiry_tours_filter' ) ) {
	/**
	 * @param $args
	 *
	 * @return array
	 */
	function inspiry_tours_filter( $args ) {

		$page_id = get_the_ID();
		$tax_query = $meta_query = array();

		// number of tours to display per page
		$number_of_tours = get_post_meta( $page_id, 'Inspiry_Tours_posts_number', true );
		if ( ! empty( $number_of_tours ) ) {
			$args['posts_per_page'] = intval( $number_of_tours );
		}

		// tour destination
		$destination = get_post_meta( $page_id, 'Inspiry_Tours_tours_destinations', false );
		if ( ! empty( $destination ) && is_array( $destination ) ) {
			$tax_query[] = array(
				'taxonomy' => 'tour-destination',
				'field'    => 'slug',
				'terms'    => $destination
			);
		}

		// tour type
		$type = get_post_meta( $page_id, 'Inspiry_Tours_tours_types', false );
		if ( ! empty( $type ) && is_array( $type ) ) {
			$tax_query[] = array(
				'taxonomy' => 'tour-type',
				'field'    => 'slug',
				'terms'    => $type
			);
		}

		// tour month
		$month = get_post_meta( $page_id, 'Inspiry_Tours_tours_months', false );
		if ( ! empty( $month ) && is_array( $month ) ) {
			$tax_query[] = array(
				'taxonomy' => 'tour-month',
				'field'    => 'slug',
				'terms'    => $month
			);
		}

		// tour maximum price
		$max_price = get_post_meta( $page_id, 'Inspiry_Tours_tours_max_price', true );
		if ( ! empty( $max_price ) ) {
			$meta_query[] = array(
				'key'     => 'inspiry_tour_price',
				'value'   => intval( $max_price ),
				'type'    => 'NUMERIC',
				'compare' => '<='
			);
		}

		if ( count( $tax_query ) > 1 ) {
			$tax_query['relation'] = 'AND';
		}

		if ( count( $tax_query ) ) {
			$args['tax_query']  = $tax_query;
		}

		if ( count( $meta_query ) ) {
			$args['meta_query'] = $meta_query;
		}
		return $args;
	}

	add_filter( 'inspiry_tours_filter', 'inspiry_tours_filter' );
}

if ( ! function_exists( 'inspiry_get_tour_meta' ) ) {
	/**
	 * Returns array of optimized tour meta values list
	 *
	 * @param string $post_id
	 *
	 * @return array
	 */
	function inspiry_get_tour_meta( $post_id = '' ) {

		if ( empty( $post_id ) ) {
			$post_id = get_the_ID();
		}

		$meta_prefix  = 'inspiry_';
		$min_meta_box = array();
		$meta_box     = get_post_custom( $post_id );

		$min_meta_box['departure']     = isset( $meta_box[ $meta_prefix . 'tour_departure' ][0] ) ? $meta_box[ $meta_prefix . 'tour_departure' ][0] : '';
		$min_meta_box['members']       = isset( $meta_box[ $meta_prefix . 'tour_members' ][0] ) ? $meta_box[ $meta_prefix . 'tour_members' ][0] : '';
		$min_meta_box['age']           = isset( $meta_box[ $meta_prefix . 'tour_min_age' ][0] ) ? $meta_box[ $meta_prefix . 'tour_min_age' ][0] : '';
		$min_meta_box['meal']          = isset( $meta_box[ $meta_prefix . 'tour_meal' ][0] ) ? $meta_box[ $meta_prefix . 'tour_meal' ][0] : '';
		$min_meta_box['accommodation'] = isset( $meta_box[ $meta_prefix . 'tour_accommodation' ][0] ) ? $meta_box[ $meta_prefix . 'tour_accommodation' ][0] : '';
		$min_meta_box['days']          = isset( $meta_box[ $meta_prefix . 'tour_days' ][0] ) ? $meta_box[ $meta_prefix . 'tour_days' ][0] : '';

		// tour meta values postfix
		if( ! empty( $min_meta_box['members'] ) ){
			$min_meta_box['members'] .= ' ' . esc_html__( 'Members', 'inspiry-tourpress' );
		}
		if( ! empty( $min_meta_box['age'] ) ){
			$min_meta_box['age'] .= esc_html__( '+', 'inspiry-tourpress' );
		}
		if( ! empty( $min_meta_box['days'] ) ){
			$min_meta_box['days'] .= ' ' . esc_html__( 'Days', 'inspiry-tourpress' );
		}


		//Returns Array of Term Names for "tour-destination"
		$destinations = wp_get_post_terms( get_the_ID(), 'tour-destination', array( 'fields' => 'names' ) );
		if ( ! empty( $destinations[0] ) ) {
			$min_meta_box['destination'] = $destinations[0];
		} else {
			$min_meta_box['destination'] = '';
		}

		//Returns Array of Term Names for "tour-month"
		$months = wp_get_post_terms( get_the_ID(), 'tour-month', array( 'fields' => 'names' ) );
		if ( ! empty( $months ) ) {

			// Added Support for Simple Custom Post Order Plugin
			( ! class_exists( 'SCPO_Engine' ) ) ? usort( $months, 'inspiry_compare_months' ) : '';

			$min_meta_box['months'] = '';
			$count = 1;

			foreach ( $months as $month ) {
				$min_meta_box['months'] .= mb_substr( $month, 0, 3 );

				if ( $count < count( $months ) && count( $months ) > 1 ) {
					$min_meta_box['months'] .= ' - ';
				}

				$count ++;
			}

		} else {
			$min_meta_box['months'] = '';
		}

		return $min_meta_box;
	}
}

if ( ! function_exists( 'inspiry_tour_metabox' ) ) {
	function inspiry_tour_metabox( $meta_boxes ) {

		$prefix = 'inspiry_';

		$meta_boxes[0]['tabs'] = array_merge( $meta_boxes[0]['tabs'], array(
			'banner' => array(
				'label' => esc_html__( 'Banner', 'inspiry-tourpress' ),
				'icon'  => 'dashicons-format-image',
			),
			'slider' => array(
				'label' => esc_html__( 'Home Slider', 'inspiry-tourpress' ),
				'icon'  => 'dashicons-slides',
			)
		) );

		$meta_boxes[0]['fields'] = array_merge( $meta_boxes[0]['fields'], array(

			// banner
			array(
				'id'   => "Inspiry_Tours_page_title",
				'name' => esc_html__( 'Banner Title', 'inspiry-tourpress' ),
				'desc' => esc_html__( 'Tour title will be displayed if empty.', 'inspiry-tourpress' ),
				'type' => 'text',
				'tab'  => 'banner',
			),
			array(
				'name'             => esc_html__( 'Banner Image', 'inspiry-tourpress' ),
				'desc'             => esc_html__( 'Recommended banner image size is 1600px by 600px. Tour featured image will be displayed if no banner is provided.', 'inspiry-tourpress' ),
				'id'               => "Inspiry_Tours_banner_image",
				'type'             => 'image_advanced',
				'max_file_uploads' => 1,
				'tab'              => 'banner',
			),

			// homepage slider
			array(
				'name'    => esc_html__( 'Add This Tour to Home Slider?', 'inspiry-tourpress' ),
				'id'      => "{$prefix}tour_in_slider",
				'type'    => 'radio',
				'std'     => '0',
				'tab'     => 'slider',
				'options' => array(
					'1' => esc_html__( 'Yes', 'inspiry-tourpress' ),
					'0' => esc_html__( 'No', 'inspiry-tourpress' )
				)
			),
			array(
				'name'             => esc_html__( 'Slide Image', 'inspiry-tourpress' ),
				'desc'             => esc_html__( 'The recommended image size is 2000px by 750px. You can use bigger or smaller image but try to keep the same height to width ratio and use exactly same size images for all slides.', 'inspiry-tourpress' ),
				'id'               => "{$prefix}tour_slider_image",
				'type'             => 'image_advanced',
				'max_file_uploads' => 1,
				'tab'              => 'slider',
			),
			array(
				'name' => esc_html__( 'Slide Smart Title', 'inspiry-tourpress' ),
				'id'   => "{$prefix}tour_smart_title",
				'type' => 'text',
				'tab'  => 'slider',
			),
			array(
				'name' => esc_html__( 'Slide Description', 'inspiry-tourpress' ),
				'id'   => "{$prefix}tour_slide_description",
				'type' => 'textarea',
				'tab'  => 'slider',
			),
		) );

		return $meta_boxes;
	}

	add_filter( 'inspiry_tour_meta_boxes', 'inspiry_tour_metabox' );
}

if ( ! function_exists( 'inspiry_tour_featured_image_size_desc' ) ) :
	function inspiry_tour_featured_image_size_desc( $content, $post_id, $thumbnail_id ){
		$current_screen = get_current_screen();
		if ( $current_screen && ( 'tour' == $current_screen->post_type ) ) {
			$help_text = '<p>' . esc_html__( 'Minimum required image size is 720px by 560px.', 'inspiry-tourpress' ) . '</p>';
			return $content . $help_text;
		}
		return $content;
	}
	add_filter( 'admin_post_thumbnail_html', 'inspiry_tour_featured_image_size_desc', 10, 3 );
endif;
