<?php
add_action( 'widgets_init', function () {
	register_widget( 'Inspiry_Recent_Posts_Widget' );
} );

class Inspiry_Recent_Posts_Widget extends WP_Widget {

	/**
	 * Sets up the widgets name etc
	 */
	public function __construct() {
		$widget_ops = array(
			'classname'   => 'inspiry_recent_posts_widget',
			'description' => esc_html__( 'Display recent blog posts.', 'inspiry-tourpress' ),
		);

		parent::__construct( 'inspiry_recent_posts_widget', esc_html__( 'Inspiry - Recent Posts', 'inspiry-tourpress' ), $widget_ops );
	}


	/**
	 * Outputs the content of the widget
	 *
	 * @param array $args
	 * @param array $instance
	 */
	public function widget( $args, $instance ) {

		// outputs the content of the widget
		echo wp_kses( $args['before_widget'], inspiry_widget_wrap_allowed_tags() );

		if ( ! empty( $instance['title'] ) ) {
			echo wp_kses(
				$args['before_title'] . apply_filters( 'widget_title', $instance['title'] ) . $args['after_title'],
				inspiry_widget_heading_allowed_tags()
			);
		}

		$number = ( ! empty( $instance['count'] ) ) ? intval( $instance['count'] ) : 3;

		?>
		<ul>
			<?php
				// WP_Query arguments
				$post_args = array(
					'post_type'           => 'post',
					'posts_per_page'      => $number,
					'meta_key'            => '_thumbnail_id',
					'compare'             => 'EXISTS',
					'ignore_sticky_posts' => 1
				);

				// The Query
				$query = new WP_Query( $post_args );

				// The Loop
				if ( $query->have_posts() ) {
					while ( $query->have_posts() ) {
						$query->the_post();

						?>
						<li class="clearfix">
							<div class="content">
								<h4><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h4>
								<span class="entry-date"><i class="fa fa-calendar-check-o"></i> <time datetime="<?php the_time( 'Y-m-d' ); ?>"><?php the_time( get_option( 'date_format' ) ); ?></time></span>
							</div>
							<?php
								if ( has_post_thumbnail() ) {
									?>
									<figure>
										<a href="<?php the_permalink(); ?>">
											<?php the_post_thumbnail( array( 78, 70 ) ); ?>
										</a>
									</figure>
									<?php
								}
							?>
						</li>
						<?php
					}

					wp_reset_postdata();

				} else {
					echo '<h4>' . esc_html__( 'No Posts Found!', 'inspiry-tourpress' ) . '</h4>';
				}
			?>
		</ul>
		<?php
		echo wp_kses( $args['after_widget'], inspiry_widget_wrap_allowed_tags() );
	}


	/**
	 * Outputs the options form on admin
	 *
	 * @param array $instance The widget options
	 *
	 * @return void
	 */
	public function form( $instance ) {
		// outputs the options form on admin
		$title   = ! empty( $instance['title'] ) ? $instance['title'] : esc_html__( 'Recent Posts', 'inspiry-tourpress' );
		$count   = ! empty( $instance['count'] ) ? $instance['count'] : esc_html__( '3', 'inspiry-tourpress' );
		$sort_by = ! empty( $instance['sort_by'] ) ? $instance['sort_by'] : esc_html__( 'recent', 'inspiry-tourpress' );
		?>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title:', 'inspiry-tourpress' ); ?></label>
			<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_html( $title ); ?>">
		</p>		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'count' ) ); ?>"><?php esc_html_e( 'Number of Posts', 'inspiry-tourpress' ); ?></label>
			<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'count' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'count' ) ); ?>" type="text" value="<?php echo esc_html( $count ); ?>"/>
		</p>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'sort_by' ) ); ?>"><?php esc_html_e( 'Sort By:', 'inspiry-tourpress' ) ?></label>
			<select name="<?php echo esc_attr( $this->get_field_name( 'sort_by' ) ); ?>" id="<?php echo esc_attr( $this->get_field_id( 'sort_by' ) ); ?>" class="widefat">
				<option value="recent"<?php selected( $sort_by, 'recent' ); ?>><?php esc_html_e( 'Most Recent', 'inspiry-tourpress' ); ?></option>
				<option value="random"<?php selected( $sort_by, 'random' ); ?>><?php esc_html_e( 'Random', 'inspiry-tourpress' ); ?></option>
			</select>
		</p>
		<?php
	}

	/**
	 * Processing widget options on save
	 *
	 * @param array $new_instance The new options
	 * @param array $old_instance The previous options
	 *
	 * @return void
	 */
	public function update( $new_instance, $old_instance ) {
		// processes widget options to be saved
		foreach ( $new_instance as $key => $value ) {
			$updated_instance[ $key ] = sanitize_text_field( $value );
		}

		return $updated_instance;
	}
}