<?php

/**
 * Testimonial custom post type class.
 */
class Inspiry_Testimonial_Post_Type {


	/**
	 * Initialize the class
	 * @since    1.0.0
	 */
	public function __construct() {

		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_styles' ) );

	}

	/**
	 * Register Testimonial Post Type
	 * @since 1.0.0
	 */
	function register_testimonial_post_type() {

		$labels = array(
			'name'                  => esc_html_x( 'Testimonials', 'Post Type General Name', 'tourpress-pack' ),
			'singular_name'         => esc_html_x( 'Testimonial', 'Post Type Singular Name', 'tourpress-pack' ),
			'menu_name'             => esc_html__( 'Testimonials', 'tourpress-pack' ),
			'name_admin_bar'        => esc_html__( 'Testimonial', 'tourpress-pack' ),
			'archives'              => esc_html__( 'Testimonial Archives', 'tourpress-pack' ),
			'attributes'            => esc_html__( 'Testimonial Attributes', 'tourpress-pack' ),
			'parent_item_colon'     => esc_html__( 'Parent Testimonial:', 'tourpress-pack' ),
			'all_items'             => esc_html__( 'All Testimonials', 'tourpress-pack' ),
			'add_new_item'          => esc_html__( 'Add New Testimonial', 'tourpress-pack' ),
			'add_new'               => esc_html__( 'Add New', 'tourpress-pack' ),
			'new_item'              => esc_html__( 'New Testimonial', 'tourpress-pack' ),
			'edit_item'             => esc_html__( 'Edit Testimonial', 'tourpress-pack' ),
			'update_item'           => esc_html__( 'Update Testimonial', 'tourpress-pack' ),
			'view_item'             => esc_html__( 'View Testimonial', 'tourpress-pack' ),
			'view_items'            => esc_html__( 'View Testimonials', 'tourpress-pack' ),
			'search_items'          => esc_html__( 'Search Testimonial', 'tourpress-pack' ),
			'not_found'             => esc_html__( 'Not found', 'tourpress-pack' ),
			'not_found_in_trash'    => esc_html__( 'Not found in Trash', 'tourpress-pack' ),
			'featured_image'        => esc_html__( 'Testimonial Image', 'tourpress-pack' ),
			'set_featured_image'    => esc_html__( 'Set testimonial image', 'tourpress-pack' ),
			'remove_featured_image' => esc_html__( 'Remove testimonial image', 'tourpress-pack' ),
			'use_featured_image'    => esc_html__( 'Use as testimonial image', 'tourpress-pack' ),
			'insert_into_item'      => esc_html__( 'Insert into testimonial', 'tourpress-pack' ),
			'uploaded_to_this_item' => esc_html__( 'Uploaded to this testimonial', 'tourpress-pack' ),
			'items_list'            => esc_html__( 'Testimonials list', 'tourpress-pack' ),
			'items_list_navigation' => esc_html__( 'Testimonials list navigation', 'tourpress-pack' ),
			'filter_items_list'     => esc_html__( 'Filter testimonials list', 'tourpress-pack' ),
		);

		$args = array(
			'label'               => esc_html__( 'Testimonial', 'tourpress-pack' ),
			'description'         => esc_html__( 'Represent testimonials of the site.', 'tourpress-pack' ),
			'labels'              => $labels,
			'supports'            => array( 'title', 'editor', 'thumbnail', ),
			'hierarchical'        => false,
			'public'              => true,
			'show_ui'             => true,
			'show_in_menu'        => true,
			'menu_position'       => 5,
			'menu_icon'           => 'dashicons-testimonial',
			'show_in_admin_bar'   => true,
			'show_in_nav_menus'   => true,
			'can_export'          => true,
			'has_archive'         => true,
			'exclude_from_search' => true,
			'publicly_queryable'  => false,
			'capability_type'     => 'post',
		);

		register_post_type( 'testimonial', $args );

	}

	/**
	 * Register meta boxes related to testimonial post type
	 *
	 * @param   array $meta_boxes
	 *
	 * @since   1.0.0
	 * @return  array   $meta_boxes
	 */
	public function register_meta_boxes( $meta_boxes ) {

		$prefix = 'inspiry_testimonial_';

		// Testimonial Meta Box
		$meta_boxes[] = array(
			'id'       => 'testimonial-meta-box',
			'title'    => esc_html__( 'Testimonial Details', 'tourpress-pack' ),
			'pages'    => array( 'testimonial' ),
			'context'  => 'normal',
			'priority' => 'high',
			'fields'   => array(
				array(
					'name' => esc_html__( 'Author Designation', 'tourpress-pack' ),
					'id'   => "{$prefix}author_desi",
					'type' => 'text',
				),
				array(
					'name' => esc_html__( 'Rating', 'tourpress-pack' ),
					'id'   => "{$prefix}rating",
					'desc' => esc_html__( 'Given rating by testimonial author.', 'tourpress-pack' ),
					'type' => 'range',
					'min'  => 1,
					'max'  => 5,
					'step' => 1,
					'std'  => 5,
				),
			)
		);

		// apply a filter before returning meta boxes
		$meta_boxes = apply_filters( 'inspiry_testimonial_meta_boxes', $meta_boxes );

		return $meta_boxes;
	}

	/**
	 * Easy Tours admin styles
	 */
	public function enqueue_admin_styles() {

		wp_enqueue_style(
			'tp-pack-admin',
			plugin_dir_url( __FILE__ ) . 'css/admin.css',
			array(),
			'1.0.0'
		);

	}

	/**
	 * Render custom column contents
	 *
	 * @param $column
	 * @param $post_id
	 *
	 * @return void
	 */
	function manage_columns_contents( $column, $post_id ) {

		switch ( $column ) {

			case 'photo':
				$photo_url = get_the_post_thumbnail_url( $post_id, 'thumbnail' );
				$photo_url = ( empty( $photo_url ) ) ? plugin_dir_url( dirname( __FILE__ ) ) . 'admin/img/placeholder-150.png'  : $photo_url;
				echo '<img src="' . esc_url( $photo_url ) . '" alt="Author" width="125" />';
				break;

			case 'designation':
				$designation = get_post_meta( $post_id, 'inspiry_testimonial_author_desi', true );
				echo ( ! empty( $designation ) ) ? esc_html( $designation ) : '<span aria-hidden="true">—</span>';
				break;

			case 'rating':
				$rating = get_post_meta( $post_id, 'inspiry_testimonial_rating', true );
				echo ( ! empty( $rating ) ) ? $this->get_rating_stars( absint( $rating ) ) : '<span aria-hidden="true">—</span>';
				break;
		}
	}

	/**
	 * @param int $rating
	 *
	 * @return string
	 */
	public function get_rating_stars( $rating = 0 ) {

		$output = '';

		$output = '<span class="inspiry-rating">';

		for ( $count = 1; $count <= $rating; $count ++ ) {
			$output .= '<span class="dashicons dashicons-star-filled"></span>';
		}

		$unrated = 5 - $rating;
		for ( $count = 1; $count <= $unrated; $count ++ ) {
			$output .= '<span class="dashicons dashicons-star-empty"></span>';
		}

		$output .= '</span>';

		return $output;
	}

	/**
	 * Update the columns list
	 *
	 * @param $columns
	 *
	 * @return array
	 */
	public function update_columns( $columns ) {

		$columns = array(
			'cb'          => '<input type="checkbox" />',
			'photo'       => esc_html__( 'Author Photo', 'tourpress-pacck' ),
			'title'       => esc_html__( 'Author Name', 'tourpress-pack' ),
			'designation' => esc_html__( 'Designation', 'tourpress-pack' ),
			'rating'      => esc_html__( 'Rating', 'tourpress-pack' ),
			'date'        => esc_html__( 'Date', 'tourpress-pack' )
		);

		return $columns;
	}

	/**
	 * Change title field placeholder
	 *
	 * @param   string $title
	 *
	 * @since   1.0.0
	 * @return  string   $title
	 */
	public function change_title_text( $title ) {
		$screen = get_current_screen();
		if ( 'testimonial' == $screen->post_type ) {
			$title = esc_html__( 'Enter testimonial author name here', 'tourpress-pack' );
		}

		return $title;
	}
}