<?php
add_action( 'widgets_init', function () {
	register_widget( 'Inspiry_Top_Rated_Tours' );
} );

class Inspiry_Top_Rated_Tours extends WP_Widget {

	/**
	 * Sets up the widgets name etc
	 */
	public function __construct() {
		$widget_ops = array(
			'classname'   => 'inspiry_top_rated_tours_widget',
			'description' => esc_html__( 'Display top rated tours.', 'easy-tours' ),
		);

		parent::__construct( 'inspiry_top_rated_tours_widget', esc_html__( 'Inspiry - Top Rated Tours', 'easy-tours' ), $widget_ops );
	}


	/**
	 * Outputs the content of the widget
	 *
	 * @param array $args
	 * @param array $instance
	 */
	public function widget( $args, $instance ) {
		// outputs the content of the widget
		echo wp_kses( $args['before_widget'], inspiry_widget_wrap_allowed_tags() );

		if ( ! empty( $instance['title'] ) ) {
			echo wp_kses(
				$args['before_title'] . apply_filters( 'widget_title', $instance['title'] ) . $args['after_title'],
				inspiry_widget_heading_allowed_tags()
			);
		}

		$number = ( empty( $instance['number'] ) ) ? 3 : $instance['number'];

		$query_args = array(
			'post_type'      => 'tour',
			'posts_per_page' => $number,
		);

		if ( 'random' == $instance['sort_by'] ) {

			$query_args['orderby'] = 'rand';

		} else if ( 'rating' == $instance['sort_by'] ) {

			$query_args['meta_key'] = 'inspiry_tour_avg_rating';
			$query_args['orderby']  = 'meta_value meta_value_num';
			$query_args['order']    = 'DESC';
		}

		if ( is_singular( 'tour' ) ) {
			$query_args['post__not_in'] = array( get_the_ID() );
		}

		$query = new WP_Query( $query_args );

		if ( $query->have_posts() ) {
			echo '<ul>';
			while ( $query->have_posts() ) {
				$query->the_post();

				$tour_meta = get_post_custom();

				if ( has_post_thumbnail() ) {
					?>
					<li class="clearfix">
						<div class="content">
							<h4><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h4>
							<?php
								inspiry_avg_rating_stars();

								// tour price
								if ( ! empty( $tour_meta['inspiry_tour_price'][0] ) ) {

									echo '<span class="offer-price">';
									if ( inspiry_has_price( true ) ) {
										echo '<i>' . inspiry_get_price() . '</i>';
									}
									inspiry_price( true, true );
									echo '</span>';
								}
							?>
						</div>
						<?php
							if ( has_post_thumbnail() ) {
								?>
								<figure>
									<a href="<?php the_permalink(); ?>">
										<?php the_post_thumbnail( array( 78, 70 ) ); ?>
									</a>
								</figure>
								<?php
							}
						?>
					</li>
					<?php

				}
			}
			echo '</ul>';
			wp_reset_postdata();
		}

		echo wp_kses( $args['after_widget'], inspiry_widget_wrap_allowed_tags() );
	}


	/**
	 * Outputs the options form on admin
	 *
	 * @param array $instance The widget options
	 *
	 * @return void
	 */
	public function form( $instance ) {

		// outputs the options form on admin
		$title   = ! empty( $instance['title'] ) ? $instance['title'] : esc_html__( 'Top Rated Tours', 'easy-tours' );
		$sort_by = ! empty( $instance['sort_by'] ) ? $instance['sort_by'] : 'rating';
		$number  = ! empty( $instance['number'] ) ? $instance['number'] : 3;
		?>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title:', 'easy-tours' ); ?></label>
			<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_html( $title ); ?>">
		</p>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'number' ) ); ?>"><?php esc_html_e( 'No Of Tours:', 'easy-tours' ); ?></label>
			<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'number' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'number' ) ); ?>" type="number" value="<?php echo intval( $number ); ?>">
		</p>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'sort_by' ) ); ?>"><?php esc_html_e( 'Sort By:', 'easy-tours' ) ?></label>
			<select name="<?php echo esc_attr( $this->get_field_name( 'sort_by' ) ); ?>" id="<?php echo esc_attr( $this->get_field_id( 'sort_by' ) ); ?>" class="widefat">
				<option value="recent"<?php selected( $sort_by, 'recent' ); ?>><?php esc_html_e( 'Most Recent', 'easy-tours' ); ?></option>
				<option value="rating"<?php selected( $sort_by, 'rating' ); ?>><?php esc_html_e( 'Rating', 'easy-tours' ); ?></option>
				<option value="random"<?php selected( $sort_by, 'random' ); ?>><?php esc_html_e( 'Random', 'easy-tours' ); ?></option>
			</select>
		</p>
		<?php
	}

	/**
	 * Processing widget options on save
	 *
	 * @param array $new_instance The new options
	 * @param array $old_instance The previous options
	 *
	 * @return void
	 */
	public function update( $new_instance, $old_instance ) {
		// processes widget options to be saved
		foreach ( $new_instance as $key => $value ) {
			$updated_instance[ $key ] = sanitize_text_field( $value );
		}

		return $updated_instance;
	}
}