<?php

/**
 * Tour custom post type class.
 */
class ET_Tour_Post_Type {

	/**
	 * Easy Tours URL slugs
	 * @access   public
	 * @var      array $options Contains URL Slugs
	 */
	public $url_slugs;

	/**
	 * Google Maps API Key
	 * @access   public
	 * @var      array $options Contains URL Slugs
	 */
	public $google_map_api_key;

	/**
	 * Initialize the class and set its settings.
	 * @since    1.0.0
	 */
	public function __construct() {

		$plugin_other_options = get_option( 'easy_tours_others_option' );
		$google_maps_api_key  = $plugin_other_options['google_maps_api_key'];

		if( ! empty( $google_maps_api_key ) ){
			$this->google_map_api_key = $google_maps_api_key;
		}

		$this->url_slugs          = get_option( 'easy_tours_url_slugs_option' );
		if ( ! isset( $this->url_slugs['tour_url_slug'] ) ) {
			update_option( 'easy_tours_url_slugs_option', $this->url_slugs_default_options() );
		}

		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_styles' ) );

	}

	/**
	 * Provides default values for url slugs options
	 */
	function url_slugs_default_options() {

		$defaults = array(
			'tour_url_slug'             => 'tour',
			'tour_type_url_slug'        => 'tour-type',
			'tour_destination_url_slug' => 'tour-destination',
			'tour_month_url_slug'       => 'tour-month',
		);

		return $defaults;

	}

	/**
	 * Register Tour Post Type
	 * @since 1.0.0
	 */
	public function register_tour_post_type() {

		$labels = array(
			'name'                  => esc_html_x( 'Tours', 'Post Type General Name', 'easy-tours' ),
			'singular_name'         => esc_html_x( 'Tour', 'Post Type Singular Name', 'easy-tours' ),
			'menu_name'             => esc_html__( 'Easy Tours', 'easy-tours' ),
			'name_admin_bar'        => esc_html__( 'Tour', 'easy-tours' ),
			'archives'              => esc_html__( 'Tour Archives', 'easy-tours' ),
			'attributes'            => esc_html__( 'Tour Attributes', 'easy-tours' ),
			'parent_item_colon'     => esc_html__( 'Parent Tour:', 'easy-tours' ),
			'all_items'             => esc_html__( 'All Tours', 'easy-tours' ),
			'add_new_item'          => esc_html__( 'Add New Tour', 'easy-tours' ),
			'add_new'               => esc_html__( 'Add New', 'easy-tours' ),
			'new_item'              => esc_html__( 'New Tour', 'easy-tours' ),
			'edit_item'             => esc_html__( 'Edit Tour', 'easy-tours' ),
			'update_item'           => esc_html__( 'Update Tour', 'easy-tours' ),
			'view_item'             => esc_html__( 'View Tour', 'easy-tours' ),
			'view_items'            => esc_html__( 'View Tours', 'easy-tours' ),
			'search_items'          => esc_html__( 'Search Tour', 'easy-tours' ),
			'not_found'             => esc_html__( 'Not found', 'easy-tours' ),
			'not_found_in_trash'    => esc_html__( 'Not found in Trash', 'easy-tours' ),
			'featured_image'        => esc_html__( 'Tour Image', 'easy-tours' ),
			'set_featured_image'    => esc_html__( 'Set tour image', 'easy-tours' ),
			'remove_featured_image' => esc_html__( 'Remove tour image', 'easy-tours' ),
			'use_featured_image'    => esc_html__( 'Use as tour image', 'easy-tours' ),
			'insert_into_item'      => esc_html__( 'Insert into tour', 'easy-tours' ),
			'uploaded_to_this_item' => esc_html__( 'Uploaded to this tour', 'easy-tours' ),
			'items_list'            => esc_html__( 'Tours list', 'easy-tours' ),
			'items_list_navigation' => esc_html__( 'Tours list navigation', 'easy-tours' ),
			'filter_items_list'     => esc_html__( 'Filter tours list', 'easy-tours' ),
		);

		$rewrite = array(
			'slug'       => apply_filters( 'et_slug_tour-destination', $this->url_slugs['tour_url_slug'] ),
			'with_front' => true,
			'pages'      => true,
			'feeds'      => true,
		);

		$args = array(
			'label'               => esc_html__( 'Tour', 'easy-tours' ),
			'description'         => esc_html__( 'Represent tours of the site.', 'easy-tours' ),
			'labels'              => $labels,
			'supports'            => array( 'title', 'editor', 'author', 'thumbnail', 'comments', 'revisions', ),
			'taxonomies'          => array( 'tour-destination', 'tour-type', 'tour-month' ),
			'hierarchical'        => false,
			'public'              => true,
			'show_ui'             => true,
			'show_in_menu'        => true,
			'menu_position'       => 5,
			'menu_icon'           => 'dashicons-palmtree',
			'show_in_admin_bar'   => true,
			'show_in_nav_menus'   => true,
			'can_export'          => true,
			'has_archive'         => true,
			'exclude_from_search' => false,
			'publicly_queryable'  => true,
			'capability_type'     => 'post',
			'rewrite'             => $rewrite,
			'show_in_rest'        => true,
			'rest_base'           => apply_filters( 'et_tours_rest_base', esc_html__( 'tours', 'easy-tours' ) )
		);

		register_post_type( apply_filters( 'et_slug_tour', 'tour' ), $args );

	}

	/**
	 * Register Tour Taxonomy Destination
	 * @since 1.0.0
	 */
	public function register_destination_taxonomy() {
		$labels = array(
			'name'                       => esc_html__( 'Destinations', 'easy-tours' ),
			'singular_name'              => esc_html__( 'Destination', 'easy-tours' ),
			'search_items'               => esc_html__( 'Search Destination', 'easy-tours' ),
			'popular_items'              => esc_html__( 'Travel Destinations', 'easy-tours' ),
			'all_items'                  => esc_html__( 'All Destinations', 'easy-tours' ),
			'parent_item'                => esc_html__( 'Parent Destination', 'easy-tours' ),
			'parent_item_colon'          => esc_html__( 'Parent Destination:', 'easy-tours' ),
			'edit_item'                  => esc_html__( 'Edit Destination', 'easy-tours' ),
			'update_item'                => esc_html__( 'Update Destination', 'easy-tours' ),
			'add_new_item'               => esc_html__( 'Add New Destination', 'easy-tours' ),
			'new_item_name'              => esc_html__( 'New Destination Name', 'easy-tours' ),
			'separate_items_with_commas' => esc_html__( 'Separate Destinations with commas', 'easy-tours' ),
			'add_or_remove_items'        => esc_html__( 'Add or Remove Destinations', 'easy-tours' ),
			'choose_from_most_used'      => esc_html__( 'Choose from the most used Destinations', 'easy-tours' ),
			'menu_name'                  => esc_html__( 'Destinations', 'easy-tours' )
		);

		$args = array(
			'labels'            => $labels,
			'hierarchical'      => true,
			'public'            => true,
			'show_ui'           => true,
			'show_admin_column' => true,
			'show_in_nav_menus' => true,
			'show_tagcloud'     => true,
			'rewrite'           => array( 'slug' => apply_filters( 'et_slug_tour-destination', $this->url_slugs['tour_destination_url_slug'] ) ),
			'show_in_rest'      => true,
			'rest_base'         => apply_filters( 'et_destinations_rest_base', esc_html__( 'destinations', 'easy-tours' ) )
		);

		register_taxonomy( 'tour-destination', array( 'tour' ), $args );
	}

	/**
	 * Register Tour Taxonomy Type
	 * @since 1.0.0
	 */
	public function register_type_taxonomy() {

		$labels = array(
			'name'                       => esc_html_x( 'Types', 'Taxonomy General Name', 'easy-tours' ),
			'singular_name'              => esc_html_x( 'Type', 'Taxonomy Singular Name', 'easy-tours' ),
			'menu_name'                  => esc_html__( 'Types', 'easy-tours' ),
			'all_items'                  => esc_html__( 'All Types', 'easy-tours' ),
			'parent_item'                => esc_html__( 'Parent Type', 'easy-tours' ),
			'parent_item_colon'          => esc_html__( 'Parent Type:', 'easy-tours' ),
			'new_item_name'              => esc_html__( 'New Type Name', 'easy-tours' ),
			'add_new_item'               => esc_html__( 'Add New Type', 'easy-tours' ),
			'edit_item'                  => esc_html__( 'Edit Type', 'easy-tours' ),
			'update_item'                => esc_html__( 'Update Type', 'easy-tours' ),
			'view_item'                  => esc_html__( 'View Type', 'easy-tours' ),
			'separate_items_with_commas' => esc_html__( 'Separate types with commas', 'easy-tours' ),
			'add_or_remove_items'        => esc_html__( 'Add or remove types', 'easy-tours' ),
			'choose_from_most_used'      => esc_html__( 'Choose from the most used', 'easy-tours' ),
			'popular_items'              => esc_html__( 'Popular Types', 'easy-tours' ),
			'search_items'               => esc_html__( 'Search Types', 'easy-tours' ),
			'not_found'                  => esc_html__( 'Not Found', 'easy-tours' ),
			'no_terms'                   => esc_html__( 'No types', 'easy-tours' ),
			'items_list'                 => esc_html__( 'Types list', 'easy-tours' ),
			'items_list_navigation'      => esc_html__( 'Types list navigation', 'easy-tours' ),
		);
		$args   = array(
			'labels'            => $labels,
			'hierarchical'      => true,
			'public'            => true,
			'show_ui'           => true,
			'show_admin_column' => true,
			'show_in_nav_menus' => true,
			'show_tagcloud'     => true,
			'rewrite'           => array( 'slug' => apply_filters( 'et_slug_tour-type', $this->url_slugs['tour_type_url_slug'] ) ),
			'show_in_rest'      => true,
			'rest_base'         => apply_filters( 'et_types_rest_base', esc_html__( 'types', 'easy-tours' ) )
		);
		register_taxonomy( 'tour-type', array( 'tour' ), $args );

	}

	/**
	 * Register Tour Taxonomy Month
	 * @since 1.0.0
	 */
	public function register_month_taxonomy() {
		$labels = array(
			'name'                       => esc_html__( 'Months', 'easy-tours' ),
			'singular_name'              => esc_html__( 'Month', 'easy-tours' ),
			'search_items'               => esc_html__( 'Search Month', 'easy-tours' ),
			'popular_items'              => esc_html__( 'Popular Months', 'easy-tours' ),
			'all_items'                  => esc_html__( 'All Months', 'easy-tours' ),
			'parent_item'                => esc_html__( 'Parent Month', 'easy-tours' ),
			'parent_item_colon'          => esc_html__( 'Parent Month:', 'easy-tours' ),
			'edit_item'                  => esc_html__( 'Edit Month', 'easy-tours' ),
			'update_item'                => esc_html__( 'Update Month', 'easy-tours' ),
			'add_new_item'               => esc_html__( 'Add New Month', 'easy-tours' ),
			'new_item_name'              => esc_html__( 'New Month Name', 'easy-tours' ),
			'separate_items_with_commas' => esc_html__( 'Separate Months with commas', 'easy-tours' ),
			'add_or_remove_items'        => esc_html__( 'Add or Remove Months', 'easy-tours' ),
			'choose_from_most_used'      => esc_html__( 'Choose from the most used Months', 'easy-tours' ),
			'menu_name'                  => esc_html__( 'Months', 'easy-tours' )
		);

		$args = array(
			'labels'            => $labels,
			'hierarchical'      => true,
			'public'            => true,
			'show_ui'           => true,
			'show_admin_column' => true,
			'show_in_nav_menus' => true,
			'show_tagcloud'     => true,
			'rewrite'           => array( 'slug' => apply_filters( 'et_slug_tour-month', $this->url_slugs['tour_month_url_slug'] ) ),
			'show_in_rest'      => true,
			'rest_base'         => apply_filters( 'et_months_rest_base', esc_html__( 'months', 'easy-tours' ) )
		);

		register_taxonomy( 'tour-month', array( 'tour' ), $args );
	}

	/**
	 * Register meta boxes related to tour post type
	 *
	 * @param   array $meta_boxes
	 *
	 * @since   1.0.0
	 * @return  array   $meta_boxes
	 */
	public function register_meta_boxes( $meta_boxes ) {

		$prefix = 'inspiry_';

		// Tour Meta Box
		$meta_boxes[] = array(
			'id'          => 'tour-meta-box',
			'title'       => esc_html__( 'Tour Details', 'easy-tours' ),
			'pages'       => array( 'tour' ),
			'context'     => 'normal',
			'priority'    => 'high',
			'tabs'        => array(
				'basic'     => array(
					'label' => esc_html__( 'Basic Info', 'easy-tours' ),
					'icon'  => 'dashicons-location',
				),
				'detail'    => array(
					'label' => esc_html__( 'Details', 'easy-tours' ),
					'icon'  => 'dashicons-clipboard',
				),
				'itinerary' => array(
					'label' => esc_html__( 'Itinerary', 'easy-tours' ),
					'icon'  => 'dashicons-list-view',
				),
				'gallery'   => array(
					'icon'  => 'dashicons-format-gallery',
					'label' => esc_html__( 'Gallery', 'easy-tours' ),
				)
			),
			'tab_style'   => 'left',
			'tab_wrapper' => true,
			'fields'      => array(

				/**
				 * Basic Tab
				 */
				array(
					'name'    => esc_html__( 'Price', 'easy-tours' ),
					'desc'    => esc_html__( 'Example: 1200', 'easy-tours' ),
					'id'      => "{$prefix}tour_price",
					'type'    => 'text',
					'tab'     => 'basic',
					'columns' => '6'
				),
				array(
					'name'    => esc_html__( 'Discounted Price', 'easy-tours' ),
					'desc'    => esc_html__( 'Example: 950', 'easy-tours' ),
					'id'      => "{$prefix}tour_price_dis",
					'type'    => 'text',
					'tab'     => 'basic',
					'columns' => '6'
				),
				array(
					'name'    => esc_html__( 'Price Per Postfix', 'easy-tours' ),
					'desc'    => esc_html__( 'Example: Person', 'easy-tours' ),
					'id'      => "{$prefix}tour_price_postfix",
					'type'    => 'text',
					'tab'     => 'basic',
					'columns' => '6'
				),
				array(
					'name'    => esc_html__( 'Departure From', 'easy-tours' ),
					'desc'    => esc_html__( 'Name of departure city', 'easy-tours' ),
					'id'      => "{$prefix}tour_departure",
					'type'    => 'text',
					'tab'     => 'basic',
					'columns' => '6'
				),
				array(
					'name'    => esc_html__( 'Number of Days', 'easy-tours' ),
					'desc'    => esc_html__( 'Example: 5', 'easy-tours' ),
					'id'      => "{$prefix}tour_days",
					'type'    => 'text',
					'tab'     => 'basic',
					'columns' => '6'
				),
				array(
					'name'    => esc_html__( 'Number of Members (Group Size)', 'easy-tours' ),
					'desc'    => esc_html__( 'Example: 10', 'easy-tours' ),
					'id'      => "{$prefix}tour_members",
					'type'    => 'text',
					'tab'     => 'basic',
					'columns' => '6'
				),
				array(
					'name'    => esc_html__( 'Minimum Age', 'easy-tours' ),
					'desc'    => esc_html__( 'Example: 18', 'easy-tours' ),
					'id'      => "{$prefix}tour_min_age",
					'type'    => 'text',
					'tab'     => 'basic',
					'columns' => '6'
				),
				array(
					'name'    => esc_html__( 'Treat this tour as featured?', 'easy-tours' ),
					'id'      => "{$prefix}tour_featured",
					'type'    => 'radio',
					'tab'     => 'basic',
					'std'     => '0',
					'options' => array(
						'1' => esc_html__( 'Yes', 'easy-tours' ),
						'0' => esc_html__( 'No', 'easy-tours' ),
					),
					'columns' => '6'
				),
				array(
					'name'    => esc_html__( 'Accommodation', 'easy-tours' ),
					'desc'    => esc_html__( 'Example: Hotel Accommodation', 'easy-tours' ),
					'id'      => "{$prefix}tour_accommodation",
					'type'    => 'text',
					'tab'     => 'basic',
					'columns' => '6'
				),
				array(
					'name'    => esc_html__( 'Meals', 'easy-tours' ),
					'desc'    => esc_html__( 'Example: Breakfast Included', 'easy-tours' ),
					'id'      => "{$prefix}tour_meal",
					'type'    => 'text',
					'tab'     => 'basic',
					'columns' => '6'
				),

				// DIVIDER
				array(
					'type' => 'divider',
					'tab'  => 'basic'
				),

				// map location section
				array(
					'name'    => esc_html__( 'Google Map Location', 'easy-tours' ),
					'id'      => "{$prefix}tour_location_display",
					'std'     => 'hide',
					'type'    => 'radio',
					'tab'     => 'basic',
					'options' => array(
						'show' => esc_html__( 'Enable', 'easy-tours' ),
						'hide' => esc_html__( 'Disable', 'easy-tours' ),
					),
					'columns' => 6,
				),
				array(
					'id'      => "{$prefix}tour_map_address",
					'name'    => esc_html__( 'Location Address', 'easy-tours' ),
					'type'    => 'text',
					'std'     => esc_html__( 'New York, NY, USA', 'easy-tours' ),
					'columns' => 6,
					'tab'     => 'basic',
				),
				array(
					'id'            => "{$prefix}tour_map_location",
					'name'          => esc_html__( 'Location on Google Map', 'easy-tours' ),
					'type'          => 'map',
					'std'           => '40.715923,-74.008924,11',
					'address_field' => "{$prefix}tour_map_address",
					'api_key'       => esc_html( $this->google_map_api_key ),
					'columns'       => 12,
					'tab'           => 'basic',
				),

				// DIVIDER
				array(
					'type' => 'divider',
					'tab'  => 'basic'
				),
				array(
					'name'             => esc_html__( 'Downloadable Brochure', 'easy-tours' ),
					'id'               => "{$prefix}tour_browser",
					'desc'             => esc_html__( 'If you have a tour related brochure, you can upload it here.', 'easy-tours' ),
					'type'             => 'file_advanced',
					'max_file_uploads' => 1,
					'tab'              => 'basic',
				),

				/*
				 * Details Tab
				 */
				array(
					'name'       => esc_html__( 'What is included!', 'easy-tours' ),
					'id'         => "{$prefix}tour_included",
					'tab'        => 'detail',
					'type'       => 'text',
					'clone'      => true,
					'sort_clone' => true,
					'columns'       => 6,
				),
				array(
					'name'       => esc_html__( 'What is not included!', 'easy-tours' ),
					'id'         => "{$prefix}tour_not_included",
					'tab'        => 'detail',
					'type'       => 'text',
					'clone'      => true,
					'sort_clone' => true,
					'columns'       => 6,
				),

				/*
				 * Gallery Tab
				 */
				array(
					'name'    => esc_html__( 'Gallery on Tour Detail Page', 'easy-tours' ),
					'id'      => "{$prefix}tour_gallery_display",
					'std'     => 'hide',
					'type'    => 'radio',
					'tab'     => 'gallery',
					'options' => array(
						'show' => esc_html__( 'Show', 'easy-tours' ),
						'hide' => esc_html__( 'Hide', 'easy-tours' ),
					),
				),
				array(
					'name'             => esc_html__( 'Gallery Images', 'easy-tours' ),
					'desc'             => esc_html__( 'It is recommended to upload bigger images like 1200px by 800px or even bigger.', 'easy-tours' ),
					'id'               => "{$prefix}tour_images",
					'type'             => 'image_advanced',
					'max_file_uploads' => 48,
					'tab'              => 'gallery',
				),

				/*
				 * Itinerary Tab
				 */
				array(
					'name'    => esc_html__( 'Itinerary on Tour Detail Page', 'easy-tours' ),
					'id'      => "{$prefix}tour_itinerary_display",
					'std'     => 'hide',
					'type'    => 'radio',
					'tab'     => 'itinerary',
					'options' => array(
						'show' => esc_html__( 'Show', 'easy-tours' ),
						'hide' => esc_html__( 'Hide', 'easy-tours' ),
					),
				),
				array(
					'name'       => esc_html__( 'Itinerary Details', 'easy-tours' ),
					'id'         => "{$prefix}tour_itinerary",
					'type'       => 'group',
					'clone'      => true,
					'sort_clone' => true,
					'tab'        => 'itinerary',
					'fields'     => array(
						array(
							'name' => esc_html__( 'Title', 'easy-tours' ),
							'id'   => "{$prefix}tour_itinerary_title",
							'std'  => '',
							'type' => 'text',
							'size' => '75'
						),
						array(
							'name' => esc_html__( 'Description', 'easy-tours' ),
							'id'   => "{$prefix}tour_itinerary_desc",
							'std'  => '',
							'type' => 'textarea',
						),
					)
				),
			)
		);

		// apply a filter before returning meta boxes
		$meta_boxes = apply_filters( 'inspiry_tour_meta_boxes', $meta_boxes );

		return $meta_boxes;

	}

	/**
	 * Provides supported icons list
	 */
	public function tour_icons_list() {

		$icons_list = array(
			'flight'    => esc_html__( 'Flight', 'easy-tours' ),
			'bed'       => esc_html__( 'Bed', 'easy-tours' ),
			'user'      => esc_html__( 'User', 'easy-tours' ),
			'clock'     => esc_html__( 'Clock', 'easy-tours' ),
			'calendar'  => esc_html__( 'Calendar', 'easy-tours' ),
			'location'  => esc_html__( 'Location', 'easy-tours' ),
			'fast-food' => esc_html__( 'Fast Food', 'easy-tours' ),
//			'skiing'    => esc_html__( 'Skiing', 'easy-tours' ),
			'group'     => esc_html__( 'Group', 'easy-tours' ),
		);

		return apply_filters( 'inspiry_tour_icon_list', $icons_list );
	}

	/**
	 * Easy Tours admin styles
	 */
	public function enqueue_admin_styles() {

		wp_enqueue_style(
			'et-meta-box',
			plugin_dir_url( __FILE__ ) . 'css/meta-box.css',
			array(),
			'1.0.0'
		);

	}

	/**
	 * Register meta boxes related to destination taxonomy
	 *
	 * @param   array $meta_boxes
	 *
	 * @since   1.0.0
	 * @return  array   $meta_boxes
	 */
	public function register_taxonomy_meta_boxes( $meta_boxes ) {
		$meta_boxes[] = array(
			'title'      => 'Destination Detail',
			'taxonomies' => 'tour-destination',
			'fields'     => array(
				array(
					'name' => esc_html__( 'Country', 'easy-tours' ),
					'id'   => 'country',
					'type' => 'text',
				),
				array(
					'name' => esc_html__( 'Lowest Temperature', 'easy-tours' ),
					'id'   => 'low_temp',
					'type' => 'text',
				),
				array(
					'name' => esc_html__( 'Highest Temperature', 'easy-tours' ),
					'id'   => 'high_temp',
					'type' => 'text',
				),
				array(
					'name' => esc_html__( 'Destination Photo', 'easy-tours' ),
					'desc'             => esc_html__( 'Recommended image size is 720px by 560px.', 'easy-tours' ),
					'id'   => 'featured_image',
					'type' => 'image_advanced',
					'max_file_uploads' => 1,
				)
			),
		);

		$meta_boxes[] = array(
			'title'      => 'Type Detail',
			'taxonomies' => 'tour-type',
			'fields'     => array(
				array(
					'name'             => esc_html__( 'Type Icon', 'easy-tours' ),
					'desc'             => esc_html__( 'Icon width and height should not be more than 100px.' , 'easy-tours' ),
					'id'               => 'type_icon',
					'type'             => 'image_advanced',
					'max_file_uploads' => 1,
				),
				array(
					'name'             => esc_html__( 'Featured Image', 'easy-tours' ),
					'desc'             => esc_html__( 'Recommended image size is 330px by 404px.', 'easy-tours' ),
					'id'               => 'featured_image',
					'type'             => 'image_advanced',
					'max_file_uploads' => 1,
				)
			),
		);

		return $meta_boxes;
	}

	/**
	 * Render custom column contents
	 *
	 * @param $column
	 * @param $post_id
	 *
	 * @return void
	 */
	function manage_columns_contents( $column, $post_id ) {

		switch ( $column ) {

			case 'photo':
				$photo_url = get_the_post_thumbnail_url( $post_id, 'thumbnail' );
				$photo_url = ( empty( $photo_url ) ) ? plugin_dir_url( dirname( __FILE__ ) ) . 'admin/img/placeholder-150.png' : $photo_url;
				echo '<img src="' . esc_url( $photo_url ) . '" alt="Featured Image" width="125" />';
				break;

			case 'price':
				echo $this->tour_price();
				break;

			case 'days':
				$days = get_post_meta( $post_id, 'inspiry_tour_days', true );
				echo ( ! empty( $days ) ) ? esc_html( $days ) : '<span aria-hidden="true">—</span>';
		}
	}

	/**
	 * Return tour price
	 *
	 * @param $format // formatted price
	 * @param $discounted //display discounted price
	 *
	 * @return mixed
	 */
	public function tour_price( $format = true, $discounted = true ) {

		$price_key    = ( $discounted ) ? 'inspiry_tour_price_dis' : 'inspiry_tour_price';
		$price_format = get_option( 'easy_tours_price_format_option' );

		$currency      = $price_format['currency_sign'];
		$currency_pos  = $price_format['currency_position'];
		$decimals      = $price_format['number_of_decimals'];
		$dec_point     = $price_format['decimal_separator'];
		$thousands_sep = $price_format['thousand_separator'];

		$amount = get_post_meta( get_the_ID(), $price_key, true );

		if ( $discounted && empty( $amount ) ) {
			$amount = get_post_meta( get_the_ID(), 'inspiry_tour_price', true );
		}

		if ( ! empty( $amount ) ) {

			if ( $format ) {

				$formatted_amount = number_format( floatval( $amount ), intval( $decimals ), esc_html( $dec_point ), esc_html( $thousands_sep ) );

				if ( $currency_pos == 'before' ) {
					$price = esc_html( $currency ) . $formatted_amount;
				} else {
					$price = $formatted_amount . esc_html( $currency );
				}
			} else {
				$price = $amount;
			}

		} else {
			$price = $price_format['empty_price_text'];
		}

		return $price;
	}

	/**
	 * Update the columns list
	 *
	 * @param $columns
	 *
	 * @return array
	 */
	public function update_columns( $columns ) {

		$columns = array(
			'cb'                        => '<input type="checkbox" />',
			'photo'                     => esc_html__( 'Tour Photo', 'easy-tours' ),
			'title'                     => esc_html__( 'Tour Name', 'easy-tours' ),
			'price'                     => esc_html__( 'Price', 'easy-tours' ),
			'days'                      => esc_html__( 'Days', 'easy-tours' ),
			'taxonomy-tour-destination' => esc_html__( 'Destinations', 'easy-tours' ),
			'taxonomy-tour-type'        => esc_html__( 'Types', 'easy-tours' ),
			'taxonomy-tour-month'       => esc_html__( 'Months', 'easy-tours' ),
			'author'                    => esc_html__( 'Author', 'easy-tours' ),
			'date'                      => esc_html__( 'Date', 'easy-tours' ),
		);

		return $columns;
	}

	/**
	 * Change title field placeholder
	 *
	 * @param   string $title
	 *
	 * @since   1.0.0
	 * @return  string   $title
	 */
	public function change_title_text( $title ) {
		$screen = get_current_screen();
		if ( 'tour' == $screen->post_type ) {
			$title = esc_html__( 'Enter tour name here', 'easy-tours' );
		}

		return $title;
	}
}