<?php

/**
 * Defines the plugin admin-specific settings.
 */
class Easy_Tours_Settings {

	/**
	 * Price format options
	 * @access   public
	 * @var      array $options Contains price format options
	 */
	public $price_format_options;

	/**
	 * URL slugs options
	 * @access   public
	 * @var      array $options Contains URL slugs options
	 */
	public $url_slugs_options;

	/**
	 * Initialize the class and set its settings.
	 * @since    1.0.0
	 */
	public function __construct() {

		$this->price_format_options = get_option( 'easy_tours_price_format_option' );
		$this->url_slugs_options    = get_option( 'easy_tours_url_slugs_option' );
		$this->others_options       = get_option( 'easy_tours_others_option' );

		add_action( 'admin_print_styles-tour_page_settings', array( $this, 'easy_tours_admin_styles' ) );

	}


	/**
	 * Easy Tours admin styles
	 */
	public function easy_tours_admin_styles() {

		wp_enqueue_style(
			'et-settings',
			plugin_dir_url( __FILE__ ) . 'css/settings.css',
			array(),
			'1.0.0'
		);
	}

	/**
	 * Add plugin settings menus
	 * @since   1.0.0
	 */
	public function easy_tours_settings_menu() {

		add_submenu_page(
			'edit.php?post_type=tour',                      // The ID of the top-level menu page to which this sub-menu item belongs
			esc_html__( 'Settings', 'easy-tours' ),         // The value used to populate the browser's title bar when the menu page is active
			esc_html__( 'Settings', 'easy-tours' ),         // The label of this sub-menu item displayed in the menu
			'administrator',                                // What roles are able to access this sub-menu item
			'settings',                                     // The ID used to represent this sub-menu item
			array(
				$this,
				'display_easy_tours_settings'
			)   // The callback function used to render the options for this sub-menu item
		);
	}

	/**
	 * Wrapper function for price format settings
	 */
	public function display_price_format_settings() {
		$this->display_easy_tours_settings( 'price_format' );
	}

	/**
	 * Wrapper function for url slugs settings
	 */
	public function display_url_slugs_settings() {
		$this->display_easy_tours_settings( 'url_slugs' );
	}

	/**
	 * Wrapper function for url slugs settings
	 */
	public function display_others_settings() {
		$this->display_easy_tours_settings( 'others' );
	}

	/**
	 * Display easy tours settings page
	 *
	 * @param   string $active_tab name of currently active tab
	 */
	public function display_easy_tours_settings( $active_tab = '' ) {
		?>
		<!-- Create a header in the default WordPress 'wrap' container -->
		<div class="wrap">

			<h2><?php esc_html_e( 'Easy Tours Settings', 'easy-tours' ); ?></h2>

			<!-- Make a call to the WordPress function for rendering errors when settings are saved. -->
			<?php settings_errors(); ?>

			<?php

				if ( isset( $_GET['tab'] ) ) {
					$active_tab = $_GET['tab'];
				} else if ( $active_tab == 'url_slugs' ) {
					$active_tab = 'url_slugs';
				} else {
					$active_tab = 'price_format';
				}
			?>

			<h2 class="nav-tab-wrapper">
				<a href="edit.php?post_type=tour&page=settings" class="nav-tab <?php echo $active_tab == 'price_format' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'Price Format', 'easy-tours' ); ?></a>
				<a href="edit.php?post_type=tour&page=settings&tab=url_slugs" class="nav-tab <?php echo $active_tab == 'url_slugs' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'URL Slugs', 'easy-tours' ); ?></a>
				<a href="edit.php?post_type=tour&page=settings&tab=others" class="nav-tab <?php echo $active_tab == 'others' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'Others', 'easy-tours' ); ?></a>
			</h2>

			<!-- Create the form that will be used to render our options -->
			<form method="post" action="options.php">
				<?php

					if ( $active_tab == 'url_slugs' ) {

						settings_fields( 'easy_tours_url_slugs_option_group' );
						do_settings_sections( 'easy_tours_url_slugs_page' );

					} else if ( $active_tab == 'others' ) {
						settings_fields( 'easy_tours_others_option_group' );
						do_settings_sections( 'easy_tours_others_page' );
					} else {

						settings_fields( 'easy_tours_price_format_option_group' );
						do_settings_sections( 'easy_tours_price_format_page' );

					}

					submit_button();
				?>
			</form>

		</div><!-- /.wrap -->
		<?php
	}

	/**
	 * Initialize price format settings options
	 */
	public function initialize_price_format_options() {

		// If the price format options do not exist then create them
		if ( false == $this->price_format_options ) {
			update_option( 'easy_tours_price_format_option', $this->price_format_default_options() );
		}

		/**
		 * Settings Section
		 */
		add_settings_section(
			'easy_tours_price_format_section',              // ID used to identify this section and with which to register options
			'',     // Title to be displayed on the administration page
			array(
				$this,
				'price_format_settings_desc'
			),   // Callback used to render the description of the section
			'easy_tours_price_format_page'                  // Page on which to add this section of options
		);

		/**
		 * Price Format Fields
		 */
		add_settings_field(
			'currency_sign',
			esc_html__( 'Currency Sign', 'easy-tours' ),
			array( $this, 'text_option_field' ),
			'easy_tours_price_format_page',
			'easy_tours_price_format_section',
			array(
				'field_id'          => 'currency_sign',
				'field_option'      => 'easy_tours_price_format_option',
				'field_default'     => '$',
				'field_description' => esc_html__( 'Default: $', 'easy-tours' ),
			)
		);

		add_settings_field(
			'currency_position',
			esc_html__( 'Currency Sign Position', 'easy-tours' ),
			array( $this, 'select_option_field' ),
			'easy_tours_price_format_page',
			'easy_tours_price_format_section',
			array(
				'field_id'          => 'currency_position',
				'field_option'      => 'easy_tours_price_format_option',
				'field_default'     => 'before',
				'field_options'     => array(
					'before' => esc_html__( 'Before ($1000)', 'easy-tours' ),
					'after'  => esc_html__( 'After (1000$)', 'easy-tours' ),
				),
				'field_description' => esc_html__( 'Default: Before', 'easy-tours' ),
			)
		);

		add_settings_field(
			'thousand_separator',
			esc_html__( 'Thousand Separator', 'easy-tours' ),
			array( $this, 'text_option_field' ),
			'easy_tours_price_format_page',
			'easy_tours_price_format_section',
			array(
				'field_id'          => 'thousand_separator',
				'field_option'      => 'easy_tours_price_format_option',
				'field_default'     => ',',
				'field_description' => esc_html__( 'Default: ,', 'easy-tours' ),
			)
		);

		add_settings_field(
			'decimal_separator',
			esc_html__( 'Decimal Separator', 'easy-tours' ),
			array( $this, 'text_option_field' ),
			'easy_tours_price_format_page',
			'easy_tours_price_format_section',
			array(
				'field_id'          => 'decimal_separator',
				'field_option'      => 'easy_tours_price_format_option',
				'field_default'     => '.',
				'field_description' => esc_html__( 'Default: .', 'easy-tours' ),
			)
		);

		add_settings_field(
			'number_of_decimals',
			esc_html__( 'Number of Decimals', 'easy-tours' ),
			array( $this, 'text_option_field' ),
			'easy_tours_price_format_page',
			'easy_tours_price_format_section',
			array(
				'field_id'          => 'number_of_decimals',
				'field_option'      => 'easy_tours_price_format_option',
				'field_default'     => '0',
				'field_description' => esc_html__( 'Default: 0', 'easy-tours' ),
			)
		);

		add_settings_field(
			'empty_price_text',
			esc_html__( 'Empty Price Text', 'easy-tours' ),
			array( $this, 'text_option_field' ),
			'easy_tours_price_format_page',
			'easy_tours_price_format_section',
			array(
				'field_id'          => 'empty_price_text',
				'field_option'      => 'easy_tours_price_format_option',
				'field_default'     => esc_html__( 'Price on call', 'easy-tours' ),
				'field_description' => esc_html__( 'Text to display in case of empty price. Example: Price on call', 'easy-tours' ),
			)
		);

		/**
		 * Register Price Format Settings
		 */
		register_setting( 'easy_tours_price_format_option_group', 'easy_tours_price_format_option' );

	}

	/**
	 * Provides default values for price format options
	 */
	function price_format_default_options() {

		$defaults = array(
			'currency_sign'      => '$',
			'currency_position'  => 'before',
			'thousand_separator' => ',',
			'decimal_separator'  => '.',
			'number_of_decimals' => '0',
			'empty_price_text'   => 'Price on call',
		);

		return $defaults;

	}

	/**
	 * Initialize url slugs settings options
	 */
	public function initialize_url_slugs_options() {

		// create plugin options if not exist
		if ( false == $this->url_slugs_options ) {
			update_option( 'easy_tours_url_slugs_option', $this->url_slugs_default_options() );
		}

		/**
		 * Settings Section
		 */
		add_settings_section(
			'easy_tours_url_slugs_section',              // ID used to identify this section and with which to register options
			'',     // Title to be displayed on the administration page
			array( $this, 'url_slugs_settings_desc' ),   // Callback used to render the description of the section
			'easy_tours_url_slugs_page'                  // Page on which to add this section of options
		);

		/**
		 * URL Slugs Fields
		 */
		add_settings_field(
			'tour_url_slug',
			esc_html__( 'Tour', 'easy-tours' ),
			array( $this, 'text_option_field' ),
			'easy_tours_url_slugs_page',
			'easy_tours_url_slugs_section',
			array(
				'field_id'          => 'tour_url_slug',
				'field_option'      => 'easy_tours_url_slugs_option',
				'field_default'     => esc_html__( 'tour', 'easy-tours' ),
				'field_description' => esc_html__( 'Default: tour', 'easy-tours' ),
			)
		);

		add_settings_field(
			'tour_destination_url_slug',
			esc_html__( 'Tour Destination', 'easy-tours' ),
			array( $this, 'text_option_field' ),
			'easy_tours_url_slugs_page',
			'easy_tours_url_slugs_section',
			array(
				'field_id'          => 'tour_destination_url_slug',
				'field_option'      => 'easy_tours_url_slugs_option',
				'field_default'     => esc_html__( 'tour-destination', 'easy-tours' ),
				'field_description' => esc_html__( 'Default: tour-destination', 'easy-tours' ),
			)
		);

		add_settings_field(
			'tour_type_url_slug',
			esc_html__( 'Tour Type', 'easy-tours' ),
			array( $this, 'text_option_field' ),
			'easy_tours_url_slugs_page',
			'easy_tours_url_slugs_section',
			array(
				'field_id'          => 'tour_type_url_slug',
				'field_option'      => 'easy_tours_url_slugs_option',
				'field_default'     => esc_html__( 'tour-type', 'easy-tours' ),
				'field_description' => esc_html__( 'Default: tour-type', 'easy-tours' ),
			)
		);

		add_settings_field(
			'tour_month_url_slug',
			esc_html__( 'Tour Month', 'easy-tours' ),
			array( $this, 'text_option_field' ),
			'easy_tours_url_slugs_page',
			'easy_tours_url_slugs_section',
			array(
				'field_id'          => 'tour_month_url_slug',
				'field_option'      => 'easy_tours_url_slugs_option',
				'field_default'     => esc_html__( 'tour-month', 'easy-tours' ),
				'field_description' => esc_html__( 'Default: tour-month', 'easy-tours' ),
			)
		);


		/**
		 * Register Settings
		 */
		register_setting( 'easy_tours_url_slugs_option_group', 'easy_tours_url_slugs_option' );

	}

	/**
	 * Provides default values for url slugs options
	 */
	function url_slugs_default_options() {

		$defaults = array(
			'tour_url_slug'             => 'tour',
			'tour_type_url_slug'        => 'tour-type',
			'tour_destination_url_slug' => 'tour-destination',
			'tour_month_url_slug'       => 'tour-month',
		);

		return $defaults;

	}

	/**
	 * Initialize url slugs settings options
	 */
	public function initialize_others_options() {

		/**
		 * Settings Section
		 */
		add_settings_section(
			'easy_tours_others_section',              // ID used to identify this section and with which to register options
			'',     // Title to be displayed on the administration page
			array( $this, 'others_settings_desc' ),   // Callback used to render the description of the section
			'easy_tours_others_page'                  // Page on which to add this section of options
		);

		/**
		 * URL Slugs Fields
		 */
		add_settings_field(
			'google_maps_api_key',
			esc_html__( 'Google Maps API Key', 'easy-tours' ),
			array( $this, 'text_option_field' ),
			'easy_tours_others_page',
			'easy_tours_others_section',
			array(
				'field_id'          => 'google_maps_api_key',
				'field_option'      => 'easy_tours_others_option',
				'field_default'     => '',
				'field_description' => wp_kses( __( 'Google Map is used for the Tours location. As of June 22, 2016 the  Google Maps no longer supports key-less access so you can get your Google Maps API Key by <a href="https://developers.google.com/maps/documentation/javascript/get-api-key" target="_blank">clicking here</a>.', 'easy-tours' ), array(
					'a' => array(
						'href' => array(),
						'target' => array(),
						'title' => array()
					)
				) ),
			)
		);

		/**
		 * Register Settings
		 */
		register_setting( 'easy_tours_others_option_group', 'easy_tours_others_option' );

	}

	/**
	 * Price format section description
	 */
	public function price_format_settings_desc() {
		echo '<p>' . esc_html__( 'Using options provided below, You can modify price format to match your needs.', 'easy-tours' ) . '</p>';
	}

	/**
	 * URL slugs section description
	 */
	public function url_slugs_settings_desc() {
		echo '<p>' . esc_html__( 'You can modify URL slugs to match your needs.', 'easy-tours' ) . '</p>';
		echo '<p class="notice update-nag notice-warning is-dismissible">' . esc_html__( 'Just make sure to re-save permalinks settings after every change in slugs to avoid 404 errors.', 'easy-tours' ) . '<br>' . esc_html__( 'You can do that from "Settings > Permalinks" page.', 'easy-tours' ) . '</p>';
	}

	/**
	 * Others section description
	 */
	public function others_settings_desc() {
		//echo '<p>' . esc_html__( 'You can modify Others settings here.', 'easy-tours' ) . '</p>';
	}

	/**
	 * Reusable text option field for settings page
	 *
	 * @param $args array   field arguments
	 */
	public function text_option_field( $args ) {
		extract( $args );
		if ( $field_id ) {

			// Default value or stored value based on option field
			if ( $field_option == 'easy_tours_url_slugs_option' ) {
				$field_value = ( isset( $this->url_slugs_options[ $field_id ] ) ) ? $this->url_slugs_options[ $field_id ] : $field_default;
			} else if ( $field_option == 'easy_tours_price_format_option' ) {
				$field_value = ( isset( $this->price_format_options[ $field_id ] ) ) ? $this->price_format_options[ $field_id ] : $field_default;
			} else {
				$field_value = ( isset( $this->others_options[ $field_id ] ) ) ? $this->others_options[ $field_id ] : $field_default;
			}

			echo '<input name="' . $field_option . '[' . $field_id . ']" class="et-text-field ' . $field_id . '" value="' . $field_value . '" />';
			if ( isset( $field_description ) ) {
				echo '<br/><label class="et-field-description">' . $field_description . '</label>';
			}

		} else {
			esc_html_e( 'Field id is missing!', 'easy-tours' );
		}
	}

	/**
	 * Reusable select options field for settings page
	 *
	 * @param $args array   field arguments
	 */
	public function select_option_field( $args ) {
		extract( $args );
		if ( $field_id ) {

			// Default value or stored value based on option field
			if ( $field_option == 'easy_tours_url_slugs_option' ) {
				$existing_value = ( isset( $this->url_slugs_options[ $field_id ] ) ) ? $this->url_slugs_options[ $field_id ] : $field_default;
			} else {
				$existing_value = ( isset( $this->price_format_options[ $field_id ] ) ) ? $this->price_format_options[ $field_id ] : $field_default;
			}

			?>
			<select name="<?php echo $field_option . '[' . $field_id . ']'; ?>" class="et-select-field <?php echo $field_id; ?>">
				<?php foreach ( $field_options as $key => $value ) { ?>
					<option value="<?php echo $key; ?>" <?php selected( $existing_value, $key ); ?>><?php echo $value; ?></option>
				<?php } ?>
			</select>
			<?php
			if ( isset( $field_description ) ) {
				echo '<br/><label class="et-field-description">' . $field_description . '</label>';
			}
		} else {
			esc_html_e( 'Field id is missing!', 'easy-tours' );
		}
	}

	/**
	 * Add plugin action links
	 *
	 * @param $links
	 *
	 * @return array
	 */
	public function easy_tours_action_links( $links ) {
		$links[] = '<a href="' . get_admin_url( null, 'edit.php?post_type=tour&page=settings' ) . '">' . esc_html__( 'Settings', 'easy-tours' ) . '</a>';

		return $links;
	}
}